<?php

namespace App\Http\Controllers;

use App\Models\Actividades;
use App\Models\AvanceModulos;
use Illuminate\Http\Request;

class ActividadesController extends Controller
{
    private $apiKey;

    function __construct() {
        $apiController = new ApiController();

        $this->apiKey = $apiController->getApiKey()["key"];
    }

    public function crearActividad(Request $request) {
        $id = new \stdClass();

        \DB::transaction(function() use(&$id, $request) {
            $actividades = Actividades::on('costos_principal');

            $id = $actividades->create(
                [
                    "con_fk_id" => $request->input("con_fk_id"),
                    "ins_fk_id" => $request->input("ins_fk_id"),
                    "act_ano" => $request->input("act_ano"),
                    "act_codigo" => $request->input("act_codigo"),
                    "act_descripcion" => $request->input("act_descripcion"),
                    "act_paquete" => $request->input("act_paquete")
                ]
            );

            $this->verificarAvance($request->input("con_fk_id"),
                                   $request->input("ins_fk_id"),
                                   $request->input("act_ano"));
        });

        return array("response" => $id->act_pk_id);
    }

    public function actualizarActividad(Request $request) {
        Actividades::on('costos_principal')
                   ->where("con_fk_id", $request->input("con_fk_id"))
                   ->where("ins_fk_id", $request->input("ins_fk_id"))
                   ->where("act_ano", $request->input("act_ano"))
                   ->where("act_codigo", $request->input("act_codigo"))
        ->update(
            [
                "act_descripcion" => $request->input("act_descripcion")
            ]
        );
    }

    public function borrarActividad(Request $request) {
        \DB::transaction(function() use($request) {
            Actividades::on('costos_principal')->where("act_pk_id", $request->input("act_pk_id"))->delete();

            $this->verificarAvance($request->input("con_fk_id"),
                                   $request->input("ins_fk_id"),
                                   $request->input("act_ano"));
        });
    }

    public function getActividades(Request $request) {
        return Actividades::on('costos_principal')
                          ->where('con_fk_id', $request->input('con_fk_id'))
                          ->where('ins_fk_id', $request->input('ins_fk_id'))
                          ->where('act_ano', $request->input('act_ano'))
                          ->orderBy('act_codigo')
                          ->orderBy('act_descripcion')
                          ->get()->toArray();
    }

    public function getActividadesCsv(Request $request) {
        return Actividades::on('costos_principal')
                          ->select('act_codigo', 'act_descripcion')
                          ->where('con_fk_id', $request->input('con_fk_id'))
                          ->where('ins_fk_id', $request->input('ins_fk_id'))
                          ->where('act_ano', $request->input('act_ano'))
                          ->orderBy('act_codigo')
                          ->orderBy('act_descripcion')
                          ->get()->toArray();
    }

    public function getActividadEspecifica(Request $request) {
        return Actividades::on('costos_principal')
                          ->where('con_fk_id', $request->input('con_fk_id'))
                          ->where('ins_fk_id', $request->input('ins_fk_id'))
                          ->where('act_ano', $request->input('act_ano'))
                          ->where('act_codigo', $request->input('act_codigo'))
                          ->get()->toArray();
    }

    public function getPaquetes(Request $request) {
        return Actividades::on('costos_principal')
                          ->where('con_fk_id', $request->input('con_fk_id'))
                          ->where('ins_fk_id', $request->input('ins_fk_id'))
                          ->where('act_ano', $request->input('act_ano'))
                          ->where('act_paquete', true)
                          ->get()->toArray();
    }

    public function getConteoPaquetes(Request $request) {
        return Actividades::on('costos_principal')
                          ->where('con_fk_id', $request->input('con_fk_id'))
                          ->where('ins_fk_id', $request->input('ins_fk_id'))
                          ->where('act_ano', $request->input('act_ano'))
                          ->where('act_paquete', true)
                          ->count();
    }

    public function cargarArchivoActividades(Request $request) {
        $apiKey = $request->apiKey;

        if ($apiKey === $this->apiKey) {
            $rutaDelArchivo = $request->actividades->path();
            $delimitador = $request->delimitador;
            $archivo = fopen($rutaDelArchivo, 'r');
            $actividades = array();
            $primeraLinea = true;

            // Extraer cada línea del archivo CSV y convertirlo en un arreglo
            while($linea = fgetcsv($archivo, 1000, $delimitador)) {
                if (!$primeraLinea) {
                    $registro = $linea;
                    array_push($actividades, $registro);
                } else {
                    $primeraLinea = false;
                }
            }

            fclose($archivo);

            return $this->procesarArchivoActividades($actividades,
                                                     $request->con_fk_id,
                                                     $request->ins_fk_id,
                                                     $request->act_ano);
        } else {
            throw new \Exception('Imposible completar la petición.');
        }
    }

    private function procesarArchivoActividades($actividades, $con_fk_id, $ins_fk_id, $act_ano) {
        $resultados = new \stdClass();
        $resultados->errores = "";

        \DB::transaction(function() use(&$resultados, $actividades, $con_fk_id, $ins_fk_id, $act_ano) {
            if (count($actividades) === 0) {
                $resultados->correcto = false;
                $resultados->errores = 'El archivo está vacío. ';
                return json_encode($resultados);
            } else if (count($actividades[0]) !== 3) {
                $resultados->correcto = false;
                $resultados->errores = 'La cantidad de columnas no corresponde. ';
                return json_encode($resultados);
            } else {
                $actividadesBd = Actividades::on('costos_principal')
                                            ->where('con_fk_id', $con_fk_id)
                                            ->where('ins_fk_id', $ins_fk_id)
                                            ->where('act_ano', $act_ano)
                                            ->get()->toArray();

                $actividadesExistentes = [];
                $registrosProcesados = 0;

                // Crear las relaciones existentes en la base de datos
                foreach ($actividadesBd as $actDb) {
                    $actividadesExistentes[$actDb['act_codigo']] = $actDb['act_descripcion'];
                }

                // Errores de los centros de utilidad
                $descripcionLargas = false;
                $camposVacios = false;

                // Revisar el archivo respecto a las relaciones existentes
                foreach($actividades as $actCsv) {
                    $codigo = strtoupper(utf8_encode(trim($actCsv[0])));
                    $descripcion = utf8_encode($actCsv[1]);
                    $paquete = strtoupper(utf8_encode($actCsv[2]));

                    if ($codigo != "" && $descripcion != "" && $paquete != "") {
                        // Descripcion largas
                        if (strlen($codigo) <= 25 && strlen($descripcion) <= 500) {
                            // Existe la actividad
                            if (isset($actividadesExistentes[$codigo])) {
                                // Actualizar si la descripcion es diferente
                                if ($actividadesExistentes[$codigo] != $descripcion) {
                                    Actividades::on('costos_principal')
                                               ->where("con_fk_id", $con_fk_id)
                                               ->where("ins_fk_id", $ins_fk_id)
                                               ->where("act_ano", $act_ano)
                                               ->where("act_codigo", $codigo)
                                    ->update(
                                        [
                                            "act_descripcion" => $descripcion
                                        ]
                                    );

                                    $registrosProcesados++;
                                } else {
                                    $registrosProcesados++;
                                }
                            } else { // No existe, es necesario crearlo
                                $actividad = Actividades::on('costos_principal');

                                $id = $actividad->create(
                                    [
                                        "con_fk_id" => $con_fk_id,
                                        "ins_fk_id" => $ins_fk_id,
                                        "act_ano" => $act_ano,
                                        "act_codigo" => $codigo,
                                        "act_descripcion" => $descripcion,
                                        "act_paquete" => $paquete == "S" ? true : false
                                    ]
                                );

                                $registrosProcesados++;

                                $actividadesExistentes[$codigo] = $descripcion;
                            }
                        } else {
                            $descripcionLargas = true;
                        }
                    } else {
                        $camposVacios = true;
                    }
                }

                // Descripciones largas
                if ($descripcionLargas) {
                    $resultados->errores .= "El codigo es incorrecto o existen descripciones con más caracteres de los permitidos.
                                             El máximo para el codigo son 25 caracteres y 500 caracteres para la descripción.";
                }

                // Campos vacios
                if ($camposVacios) {
                    if ($resultados->errores !== "") {
                        $resultados->errores .= "-  ";
                    }

                    $resultados->errores .= "Hay campos vacíos.";
                }

                if (!$descripcionLargas && !$camposVacios) {
                    $resultados->correcto = true;
                } else {
                    $resultados->correcto = false;
                }

                $resultados->registros = $registrosProcesados;
            }

            $this->verificarAvance($con_fk_id, $ins_fk_id, $act_ano);
        });

        return json_encode($resultados);
    }

    public function verificarAvance($contrato, $institucion, $ano) {
        // Verificar si existe registro del avance
        $conteo = AvanceModulos::on('costos_principal')
                               ->where('con_fk_id', $contrato)
                               ->where('ins_fk_id', $institucion)
                               ->where('avm_ano', $ano)
                               ->whereNull('avm_mes')
                               ->count();

        // Verificar el avance del modulo
        $act = Actividades::on('costos_principal')
                          ->where('con_fk_id', $contrato) 
                          ->where('ins_fk_id', $institucion)
                          ->where('act_ano', $ano)
                          ->count();

        if ($conteo > 0) {
            if ($act > 0) {
                AvanceModulos::on('costos_principal')
                                ->where('con_fk_id', $contrato)
                                ->where('ins_fk_id', $institucion)
                                ->where('avm_ano', $ano)
                                ->where('avm_mes', null)
                ->update(
                    [
                        "avm_actividades" => true
                    ]
                );
            } else {
                AvanceModulos::on('costos_principal')
                            ->where('con_fk_id', $contrato)
                            ->where('ins_fk_id', $institucion)
                            ->where('avm_ano', $ano)
                            ->where('avm_mes', null)
                ->update(
                    [
                        "avm_actividades" => false
                    ]
                ); 
            }
        } else {
            AvanceModulos::on('costos_principal')
            ->create(
                [
                    "con_fk_id" => $contrato,
                    "ins_fk_id" => $institucion,
                    "avm_ano" => $ano,
                    "avm_mes" => null,
                    "avm_actividades" => true
                ]
            );
        }          
    }
}
